<?php

namespace App\Http\Controllers;

use App\Models\Availability;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Validator;

class AvailabilityController extends Controller
{
    public function index(): JsonResponse
    {
        $availabilities = Availability::with(['candidateProfile'])->get();
        return response()->json([
            'status' => 'success',
            'data' => $availabilities,
        ], 200);
    }

    public function show($profileId): JsonResponse
    {
        $availability = Availability::where('profile_id', $profileId)->first();

        if (!$availability) {
            return response()->json([
                'status' => 'error',
                'message' => 'Availability not found for this profile',
            ], 404);
        }

        return response()->json([
            'status' => 'success',
            'data' => $availability,
        ], 200);
    }

    public function store(Request $request): JsonResponse
    {
        // Disable store method, use PUT instead
        return response()->json([
            'status' => 'error',
            'message' => 'Use PUT request to /api/availabilities/{profileId} for create/update operations.',
        ], 405); // Method Not Allowed
    }

    public function update(Request $request, $profileId): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'availability' => 'sometimes|array',
            'availability.*.morning' => 'sometimes|boolean',
            'availability.*.afternoon' => 'sometimes|boolean',
            'availability.*.evening' => 'sometimes|boolean',
            'timezone' => 'sometimes|string|max:50',
        ], [
            'availability.*.*.boolean' => 'Each time slot must be true or false.',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        // Fetch or initialize availability for the profile
        $availability = Availability::where('profile_id', $profileId)->first();

        // Default 7-day structure
        $days = ['monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday', 'sunday'];
        $defaultAvailability = [];
        foreach ($days as $day) {
            $defaultAvailability[$day] = [
                'morning' => false,
                'afternoon' => false,
                'evening' => false,
            ];
        }

        $availabilityData = $defaultAvailability;

        if ($availability) {
            // If record exists, use existing availability as base
            $availabilityData = $availability->availability;
        }

        // Merge provided availability with default/existing data
        if ($request->has('availability')) {
            foreach ($request->availability as $day => $slots) {
                if (isset($availabilityData[$day])) {
                    $availabilityData[$day] = array_merge($availabilityData[$day], $slots);
                } else {
                    $availabilityData[$day] = $slots;
                }
            }
        }

        $timezone = $request->timezone ?? ($availability ? $availability->timezone : 'America/Toronto');

        if ($availability) {
            $availability->update([
                'availability' => $availabilityData,
                'timezone' => $timezone,
            ]);
            $responseData = $availability;
        } else {
            $responseData = Availability::create([
                'profile_id' => $profileId,
                'availability' => $availabilityData,
                'timezone' => $timezone,
            ]);
        }

        return response()->json([
            'status' => 'success',
            'data' => $responseData,
        ], $availability ? 200 : 201); // 201 for create, 200 for update
    }

    public function destroy($profileId): JsonResponse
    {
        $availability = Availability::where('profile_id', $profileId)->first();

        if (!$availability) {
            return response()->json([
                'status' => 'error',
                'message' => 'Availability not found for this profile',
            ], 404);
        }

        $availability->delete();

        return response()->json([
            'status' => 'success',
            'message' => 'Availability deleted successfully',
        ], 200);
    }
}
